const database = require('./database');
const translator = require('./translator');

class DepartmentCommands {
  
  /**
   * Enable/Disable departments system
   */
  async toggleDepartments(adminPhone) {
    const currentStatus = database.getSetting('departments_enabled');
    const newStatus = currentStatus === '1' ? '0' : '1';
    
    database.setSetting('departments_enabled', newStatus);
    
    const statusText = newStatus === '1' ? '✅ ENABLED' : '❌ DISABLED';
    
    return {
      success: true,
      response: `🏢 *Departments System ${statusText}*\n\n${newStatus === '1' ? 
        '✅ Clients will now select a department when creating tickets.\n✅ Agents will be notified based on their departments.' : 
        '⚠️ System disabled. Tickets will be created normally without departments.'}`
    };
  }

  /**
   * Create a new department
   */
  async createDepartment(adminPhone, args) {
    if (!args || args.length === 0) {
      return {
        success: false,
        response: `❌ *Usage:* /create department [name]\n\n*Example:*\n/create department Technical Support`
      };
    }

    const name = args.join(' ');
    const departmentId = database.createDepartment(name, '', 0);
    
    if (!departmentId) {
      return {
        success: false,
        response: '❌ Error creating department. Name may already exist.'
      };
    }

    return {
      success: true,
      response: `✅ *Department Created!*\n\n🏢 *Name:* ${name}\n🆔 *ID:* ${departmentId}\n\n💡 Use */add agent to department* to assign agents.`
    };
  }

  /**
   * List all departments
   */
  async listDepartments(adminPhone) {
    const departments = database.listDepartments();
    
    if (departments.length === 0) {
      return {
        success: true,
        response: `📋 *Departments*\n\n⚠️ No departments registered yet.\n\n💡 Use */create department [name]* to create one.`
      };
    }

    let response = `🏢 *Departments List*\n\n`;
    
    for (const dept of departments) {
      const agents = database.getDepartmentAgents(dept.id);
      const generalBadge = dept.is_general ? ' 🌟 GENERAL' : '';
      const statusBadge = dept.status === 'active' ? '✅' : '❌';
      
      response += `${statusBadge} *${dept.id}. ${dept.name}*${generalBadge}\n`;
      if (dept.description) {
        response += `   📝 ${dept.description}\n`;
      }
      response += `   👥 Agents: ${agents.length}\n`;
      response += `   📅 Created: ${new Date(dept.created_at).toLocaleDateString()}\n\n`;
    }

    response += `\n💡 *Commands:*\n`;
    response += `• */department info [ID]* - View details\n`;
    response += `• */edit department [ID]* - Edit department\n`;
    response += `• */delete department [ID]* - Delete department`;

    return {
      success: true,
      response: response
    };
  }

  /**
   * Get department info
   */
  async getDepartmentInfo(adminPhone, departmentId) {
    if (!departmentId) {
      return {
        success: false,
        response: '❌ *Usage:* /department info [ID]'
      };
    }

    const dept = database.getDepartment(departmentId);
    
    if (!dept) {
      return {
        success: false,
        response: `❌ Department #${departmentId} not found.`
      };
    }

    const agents = database.getDepartmentAgents(departmentId);
    const generalBadge = dept.is_general ? ' 🌟 GENERAL' : '';
    const statusBadge = dept.status === 'active' ? '✅ Active' : '❌ Inactive';

    let response = `🏢 *Department #${dept.id}*${generalBadge}\n\n`;
    response += `📌 *Name:* ${dept.name}\n`;
    response += `📊 *Status:* ${statusBadge}\n`;
    
    if (dept.description) {
      response += `📝 *Description:* ${dept.description}\n`;
    }
    
    response += `\n👥 *Agents (${agents.length}):*\n`;
    
    if (agents.length === 0) {
      response += `   ⚠️ No agents assigned\n`;
    } else {
      for (const agent of agents) {
        response += `   • ${agent.name} (${agent.phone})\n`;
      }
    }

    response += `\n📅 *Created:* ${new Date(dept.created_at).toLocaleString()}\n`;
    
    return {
      success: true,
      response: response
    };
  }

  /**
   * Add agent to department
   */
  async addAgentToDepartment(adminPhone, args) {
    const adminCommands = require('./adminCommands');
    
    // Start flow to select agent and department
    adminCommands.adminStates.set(adminPhone, {
      flow: 'awaiting_agent_phone_for_department'
    });
    
    return {
      success: true,
      response: `👥 *Add Agent to Department*\n\nPlease send the agent's phone number (e.g., 258840000000)`,
      action: 'awaiting_agent_phone_for_department'
    };
  }

  /**
   * Process agent phone for department assignment
   */
  async processAgentPhoneForDepartment(adminPhone, agentPhone) {
    console.log(`📞 Processing agent phone for department: ${agentPhone}`);
    
    // Validate agent exists
    const agent = database.getAgent(agentPhone);
    
    if (!agent) {
      return {
        success: false,
        response: `❌ Agent ${agentPhone} not found.\n\n💡 Use */list agents* to see all agents.`
      };
    }

    // List departments
    const departments = database.listDepartments(true);
    
    if (departments.length === 0) {
      return {
        success: false,
        response: `❌ No departments available.\n\n💡 Create a department first using */create department [name]*`
      };
    }

    let response = `✅ *Agent Selected:* ${agent.name}\n\n`;
    response += `🏢 *Select Department:*\n\n`;
    
    departments.forEach((dept, index) => {
      const generalBadge = dept.is_general ? ' 🌟' : '';
      response += `*${index + 1}.* ${dept.name}${generalBadge}\n`;
    });
    
    response += `\n💡 Send the department number (1-${departments.length})`;

    return {
      success: true,
      response: response,
      action: 'awaiting_department_selection',
      agentPhone: agentPhone,
      departments: departments
    };
  }

  /**
   * Confirm agent to department assignment
   */
  async confirmAgentToDepartment(adminPhone, agentPhone, departmentId) {
    const success = database.addAgentToDepartment(agentPhone, departmentId);
    
    if (!success) {
      return {
        success: false,
        response: `❌ Agent already assigned to this department or error occurred.`
      };
    }

    const agent = database.getAgent(agentPhone);
    const dept = database.getDepartment(departmentId);
    
    return {
      success: true,
      response: `✅ *Agent Added to Department!*\n\n👤 *Agent:* ${agent.name}\n🏢 *Department:* ${dept.name}\n\n💡 The agent will now receive notifications for tickets in this department.`
    };
  }

  /**
   * Remove agent from department
   */
  async removeAgentFromDepartment(adminPhone, args) {
    const adminCommands = require('./adminCommands');
    
    adminCommands.adminStates.set(adminPhone, {
      flow: 'awaiting_agent_phone_for_removal'
    });
    
    return {
      success: true,
      response: `👥 *Remove Agent from Department*\n\nPlease send the agent's phone number`,
      action: 'awaiting_agent_phone_for_removal'
    };
  }

  /**
   * Process agent removal from department
   */
  async processAgentRemovalFromDepartment(adminPhone, agentPhone) {
    const agent = database.getAgent(agentPhone);
    
    if (!agent) {
      return {
        success: false,
        response: `❌ Agent ${agentPhone} not found.`
      };
    }

    const departments = database.getAgentDepartments(agentPhone);
    
    if (departments.length === 0) {
      return {
        success: false,
        response: `❌ Agent ${agent.name} is not assigned to any department.`
      };
    }

    let response = `✅ *Agent:* ${agent.name}\n\n`;
    response += `🏢 *Select Department to Remove:*\n\n`;
    
    departments.forEach((dept, index) => {
      const generalBadge = dept.is_general ? ' 🌟' : '';
      response += `*${index + 1}.* ${dept.name}${generalBadge}\n`;
    });
    
    response += `\n💡 Send the department number (1-${departments.length})`;

    return {
      success: true,
      response: response,
      action: 'awaiting_department_removal',
      agentPhone: agentPhone,
      departments: departments
    };
  }

  /**
   * Confirm agent removal from department
   */
  async confirmAgentRemoval(adminPhone, agentPhone, departmentId) {
    const success = database.removeAgentFromDepartment(agentPhone, departmentId);
    
    if (!success) {
      return {
        success: false,
        response: `❌ Error removing agent from department.`
      };
    }

    const agent = database.getAgent(agentPhone);
    const dept = database.getDepartment(departmentId);
    
    return {
      success: true,
      response: `✅ *Agent Removed!*\n\n👤 *Agent:* ${agent.name}\n🏢 *Department:* ${dept.name}\n\n💡 Agent will no longer receive notifications from this department.`
    };
  }

  /**
   * Set department as general
   */
  async setGeneralDepartment(adminPhone, departmentId) {
    if (!departmentId) {
      return {
        success: false,
        response: '❌ *Usage:* /set general department [ID]'
      };
    }

    const dept = database.getDepartment(departmentId);
    
    if (!dept) {
      return {
        success: false,
        response: `❌ Department #${departmentId} not found.`
      };
    }

    // Remove general status from all departments
    const allDepts = database.listDepartments();
    for (const d of allDepts) {
      if (d.is_general) {
        database.updateDepartment(d.id, { is_general: 0 });
      }
    }

    // Set new general department
    database.updateDepartment(departmentId, { is_general: 1 });

    return {
      success: true,
      response: `✅ *General Department Set!*\n\n🌟 *Department:* ${dept.name}\n\n💡 This department will receive ALL ticket notifications, regardless of client selection.`
    };
  }

  /**
   * Delete department
   */
  async deleteDepartment(adminPhone, departmentId) {
    if (!departmentId) {
      return {
        success: false,
        response: '❌ *Usage:* /delete department [ID]'
      };
    }

    const dept = database.getDepartment(departmentId);
    
    if (!dept) {
      return {
        success: false,
        response: `❌ Department #${departmentId} not found.`
      };
    }

    // Check if has active tickets
    const tickets = database.db.prepare(`
      SELECT COUNT(*) as count 
      FROM tickets 
      WHERE department_id = ? AND status IN ('open', 'assigned')
    `).get(departmentId);

    if (tickets.count > 0) {
      return {
        success: false,
        response: `❌ Cannot delete department with active tickets.\n\n📋 Active tickets: ${tickets.count}\n\n💡 Close all tickets first or transfer them to another department.`
      };
    }

    database.deleteDepartment(departmentId);

    return {
      success: true,
      response: `✅ *Department Deleted!*\n\n🏢 ${dept.name} has been removed from the system.`
    };
  }

  /**
   * Edit department
   */
  async editDepartment(adminPhone, departmentId) {
    if (!departmentId) {
      return {
        success: false,
        response: '❌ *Usage:* /edit department [ID]'
      };
    }

    const dept = database.getDepartment(departmentId);
    
    if (!dept) {
      return {
        success: false,
        response: `❌ Department #${departmentId} not found.`
      };
    }

    const adminCommands = require('./adminCommands');
    
    adminCommands.adminStates.set(adminPhone, {
      flow: 'awaiting_department_edit',
      departmentId: departmentId
    });

    return {
      success: true,
      response: `✏️ *Edit Department #${dept.id}*\n\n📌 *Current Name:* ${dept.name}\n\n💡 Send the new department name or "cancel" to abort.`,
      action: 'awaiting_department_edit',
      departmentId: departmentId
    };
  }

  /**
   * Process department edit
   */
  async processDepartmentEdit(adminPhone, departmentId, newName) {
    if (newName.toLowerCase() === 'cancel') {
      return {
        success: true,
        response: '❌ Edit cancelled.'
      };
    }

    database.updateDepartment(departmentId, { name: newName });

    return {
      success: true,
      response: `✅ *Department Updated!*\n\n🏢 *New Name:* ${newName}\n🆔 *ID:* ${departmentId}`
    };
  }
}

module.exports = new DepartmentCommands();
